/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx__sleep.h"
#include "mx__mcp_request_ring.h"
#  include "mx_stbar.h"
#  include "mx_byteswap.h"
#include "mx__lib.h"
#include "mx__requests.h"
#include "mx__request.h"
#include "mx__shmem.h"
#include "mx__driver_interface.h"
#include "mx__handle_map.h"
#include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__wait_queue.h"

void mx__uwake_req(struct mx_endpoint *ep)
{
#if MX_USE_SHMEM
  uint32_t type, read_idx;
#endif
  if (ep->wake_pending == 0) {
    mcp_ureq_t *req_user;

    ep->wake_mcp_handle = mx__endpoint_alloc_mcp_handle(ep);
    req_user = MX__UREQ(ep, ep->wake_mcp_handle);
    mx__post_ureq_wake(ep->is_ze, req_user, ep->eventq_flow);
    ep->wake_pending = 1;
  }
#if MX_USE_SHMEM
  if (mx__opt.disable_shmem)
    return;
  ep->shm->shmq->waiting = 1;
  MX_READBAR();
  read_idx = ep->shm->shmq->read_idx;
  type = ep->shm->shmq->queue[MX__SHM_REQ_SLOT(read_idx)].type;
  if (!((type ^ read_idx) & MX__SHM_REQ_CNT)) {
    /* rather than trying to avoid a race (between sleeping and event
       arriving in shm queue), it is easier to detect it here, and do
       as if it didn't happen (act like if the sender actually arrived
       a few nanoseconds later, and do the wake for him */
    if (ep->app_waiting)
      mx__app_wake(ep->handle);
    mx__wake(ep->handle);
  }
#endif
}

void mx__sleep_on_wait_queue(struct mx_endpoint *endpoint, struct mx__wait_queue *wq,
			     uint32_t timeout, int *waiters, struct mx__wait_queue_head *wq_head)
{
  struct timeval start,cur;
  int stimeout = timeout * 1000;
  mx_wait_t x;
  int ret;

  wq->wait_state = MX__WAIT_STATE_WAITING;
  if (wq_head) {
    mx__wait_queue_enqueue (wq_head, wq);
  }
  (*waiters)++;
  MX__EVENT_INIT(&wq->event);
  if (MX_THREAD_SAFE && !mx__opt.monothread) {
    if (endpoint->app_waiting) {
      mx__uwake_req(endpoint);
      if (wq->wait_state == MX__WAIT_STATE_WAITING) {
	if (timeout == MX_INFINITE) {
	  MX__EVENT_WAIT(&wq->event, &endpoint->lock);
	}
	else {
	  MX__EVENT_WAIT_TIMEOUT(&wq->event, &endpoint->lock, timeout);
	}
      }
    } else {
      start.tv_sec = 0;
      start.tv_usec = 0;

      while (wq->wait_state == MX__WAIT_STATE_WAITING) {
	if (timeout == MX_INFINITE) {
	  x.timeout = MX_MAX_WAIT;
	} else {
	  mx_gettimeofday(&cur, NULL);
	  if (!start.tv_sec)
	    start = cur;
	  stimeout -= (cur.tv_sec - start.tv_sec) * 1000000 + (cur.tv_usec - start.tv_usec);
	  x.timeout = stimeout / 1000;
	  if ((int)x.timeout <= 0)
	    break;
	}
	mx__uwake_req(endpoint);
	x.status = 0;
	x.mcp_wake_events = 0;
	endpoint->app_waiting = 1;
	MX__MUTEX_UNLOCK(&endpoint->lock);
	ret = mx__app_wait(endpoint->handle, &x);
	if (ret != 0) {
	  mx_fatal("mx__app_wait failed, check kernel logs for error messages");
	}
	MX__MUTEX_LOCK(&endpoint->lock);
	endpoint->app_waiting = 0;
	mx_assert(endpoint->wake_pending >= x.mcp_wake_events);
	if (x.mcp_wake_events) {
	  endpoint->wake_pending -= x.mcp_wake_events;
	  mx__endpoint_free_mcp_handle(endpoint, endpoint->wake_mcp_handle);
	}
	mx_check_wait_status(endpoint, &x.status);
	mx__luigi(endpoint);
	/* If another thread is blocked, submit a wake request
	   so the progression thread will be awoken in a timely
	   fashion to wake the blocked thread */
	if (wq->wait_state != MX__WAIT_STATE_WAITING &&
	    ((endpoint->wait_waiters > 0 ||
	      endpoint->peek_waiters > 0 || endpoint->probe_waiters > 0)))
	  mx__uwake_req(endpoint);
#ifdef MX_KERNEL
	/* kernel lib MUST be interruptible */
	if (x.status != MX_WAIT_STATUS_GOOD)
	  break;
#endif
      }
    }
  } else {
#ifndef MX_KERNEL
    if (timeout == MX_INFINITE) {
      while (wq->wait_state == MX__WAIT_STATE_WAITING) {
	mx__luigi(endpoint);
	mx_sched_yield();
      }
    } else {
      start.tv_sec = 0;
      start.tv_usec = 0;
      while (wq->wait_state == MX__WAIT_STATE_WAITING && stimeout > 0) {
	mx_gettimeofday(&cur, NULL);
	if (!start.tv_sec)
	  start = cur;
	mx_sched_yield();
	stimeout -= (cur.tv_sec - start.tv_sec) * 1000000 + (cur.tv_usec - start.tv_usec);
	start = cur;
	mx__luigi(endpoint);
      }
    }
#else
    mx_fatal("kernel always has a thread");
#endif
  }
  MX__EVENT_DESTROY(&wq->event);
  if (wq->wait_state == MX__WAIT_STATE_WAITING) {
    wq->wait_state = MX__WAIT_STATE_COMPLETE;
    waiters[0]--;
    if (wq_head) {
      mx__wait_queue_spliceout(wq_head, wq);
    }
  }
}
